<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Twilio\Rest\Client;
use App\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Crypt;

class CustomLoginController extends Controller
{
    private $otpDir;
    private $config;

    public function __construct()
    {
        $this->middleware('guest');
        $this->config = Config::first();
        $this->otpDir = public_path('otp');

        if (!File::exists($this->otpDir)) {
            File::makeDirectory($this->otpDir, 0755, true);
        }
    }

    private function isTwilioEnabled()
    {
        return $this->config->otp_enable ?? false;
    }

    private function generateOTP()
    {
        return rand(100000, 999999);
    }

    private function storeOTPAndData($phone, $otp, $data = null, $isRegistration = false)
    {
        $filename = $this->otpDir . '/' . $phone . '.txt';
        $content = $otp . "\n";

        if ($data) {
            if ($isRegistration) {
                // Encrypt data for registration
                $content .= Crypt::encrypt(json_encode($data)) . "\n";
            } else {
                // Store data without encryption for login
                $content .= json_encode($data) . "\n";
            }
        } else {
            $content .= "\n"; // Empty line if no data
        }

        $expirationTime = time() + 300; // 5 minutes from now
        $content .= $expirationTime;
        File::put($filename, $content);

        $this->startBackgroundDeletion($filename, $expirationTime);
    }

    private function startBackgroundDeletion($filename, $expirationTime)
    {
        if (function_exists('exec') && !in_array('exec', explode(',', ini_get('disable_functions')))) {
            $command = "php -r \"sleep(" . ($expirationTime - time()) . "); " .
                       "if (time() >= $expirationTime && file_exists('$filename')) { " .
                       "unlink('$filename'); }\" > /dev/null 2>&1 & echo $!";
            exec($command);
        }
        // If exec() is not available, we don't perform any action
        // The file will remain and can be cleaned up later if needed
    }

    private function getOTPAndData($phone)
    {
        $filename = $this->otpDir . '/' . $phone . '.txt';
        if (File::exists($filename)) {
            $content = File::get($filename);
            $lines = explode("\n", trim($content));

            $otp = $lines[0]; // First line is the OTP
            $data = null;
            $expirationTime = null;

            if (count($lines) > 1 && !empty($lines[1])) {
                try {
                    // Attempt to decrypt (for registration data)
                    $data = json_decode(Crypt::decrypt($lines[1]), true);
                } catch (\Illuminate\Contracts\Encryption\DecryptException $e) {
                    // If decryption fails, assume it's unencrypted login data
                    $data = json_decode($lines[1], true);
                }
            }
            if (count($lines) > 2) {
                $expirationTime = (int)$lines[2]; // Third line is expiration timestamp
            }

            // Check if OTP has expired
            if ($expirationTime && time() > $expirationTime) {
                return null;
            }

            return [
                'otp' => $otp,
                'data' => $data
            ];
        }

        return null;
    }

    private function deleteOTPFile($phone)
    {
        $filename = $this->otpDir . '/' . $phone . '.txt';
        if (File::exists($filename)) {
            File::delete($filename);
        }
    }

    public function showOTPVerificationForm()
    {
        $phone = session('phone');
        if (!$phone) {
            return redirect()->route('login');
        }
        return view('auth.otp_verification', compact('phone'));
    }

    public function showLoginForm()
    {
        $twilioEnabled = $this->isTwilioEnabled();
        return view('auth.login', compact('twilioEnabled'));
    }

    public function showRegisterForm()
    {
        return view('auth.register');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();
            return redirect()->intended('/')->with('success', 'Logged in successfully!');
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ]);
    }

    public function initiateRegistration(Request $request)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
            'full_mobile' => 'required|string|unique:users,mobile',
        ];

        if ($this->config->captcha == 1) {
            $rules['g-recaptcha-response'] = 'required|captcha';
        }

        $request->validate($rules);

        if ($this->isTwilioEnabled()) {
            $otp = $this->generateOTP();

            try {
                $accountSid = env('TWILIO_SID');
                $authToken = env('TWILIO_AUTH_TOKEN');
                $twilioNumber = env('TWILIO_NUMBER');

                $client = new Client($accountSid, $authToken);
                $client->messages->create(
                    $request->full_mobile,
                    [
                        'from' => $twilioNumber,
                        'body' => "Your OTP for registration is: $otp"
                    ]
                );

                $this->storeOTPAndData($request->full_mobile, $otp, $request->only(['name', 'email', 'password', 'full_mobile']), true);

                return redirect()->route('verify.otp.form')->with('phone', $request->full_mobile);
            } catch (\Exception $e) {
                return back()->withErrors(['error' => 'Failed to send OTP: ' . $e->getMessage()]);
            }
        } else {
            // Register user directly without OTP verification
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'mobile' => $request->full_mobile,
                'status' => 1,
                'is_admin' => 0,
                'email_verified_at' => now(),
            ]);

            Auth::login($user);
            return redirect('/')->with('success', 'Registration successful!');
        }
    }

    public function resendOTP(Request $request)
    {
        $phone = $request->input('phone');
    
        if (!$phone) {
            return response()->json(['error' => 'No phone number found for OTP resend.'], 400);
        }
    
        $storedData = $this->getOTPAndData($phone);
    
        $newOtp = $this->generateOTP();
    
        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');
    
            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $phone,
                [
                    'from' => $twilioNumber,
                    'body' => "Your new OTP is: $newOtp"
                ]
            );
    
            $isRegistration = $storedData && isset($storedData['data']['name']);
            $this->storeOTPAndData($phone, $newOtp, $storedData ? $storedData['data'] : null, $isRegistration);
    
            return response()->json(['success' => 'New OTP sent.']);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to resend OTP: ' . $e->getMessage()], 500);
        }
    }

    public function verifyOTP(Request $request)
    {
        $request->validate([
            'otp' => 'required|numeric',
            'phone' => 'required|string',
        ]);

        $storedData = $this->getOTPAndData($request->phone);

        if (!$storedData) {
            return back()->withErrors(['otp' => 'OTP has expired or is invalid']);
        }

        if ($storedData['otp'] !== $request->otp) {
            return back()->withErrors(['otp' => 'Invalid OTP']);
        }

        // OTP is valid, proceed with registration or login
        if ($storedData['data'] && isset($storedData['data']['name'])) {
            $userData = $storedData['data'];

            // Register new user
            $user = User::create([
                'name' => $userData['name'],
                'email' => $userData['email'],
                'password' => Hash::make($userData['password']),
                'mobile' => $userData['full_mobile'],
                'status' => 1,
                'is_admin' => 0, // Set default value for is_admin
                'email_verified_at' => now(), // Set email as verified
            ]);

            $this->deleteOTPFile($request->phone);

            Auth::login($user);
            return redirect()->intended('/')->with('success', 'Registration successful!');
        } else {
            $user = User::where('mobile', $request->phone)->first();
            if (!$user) {
                return back()->withErrors(['error' => 'User not found']);
            }

            Auth::login($user);
            $this->deleteOTPFile($request->phone);

            return redirect()->intended('/')->with('success', 'Logged in successfully!');
        }
    }
  
  
    public function initiateOTPLogin(Request $request)
    {
        $request->validate([
            'full_mobile' => 'required|string',
        ]);

        $user = User::where('mobile', $request->full_mobile)->first();

        if (!$user) {
            return back()->withErrors(['error' => 'No account found with this phone number.']);
        }

        $otp = $this->generateOTP();

        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');

            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $request->full_mobile,
                [
                    'from' => $twilioNumber,
                    'body' => "Your OTP for login is: $otp"
                ]
            );

            $this->storeOTPAndData($request->full_mobile, $otp, null, false);

            return redirect()->route('verify.otp.form')->with('phone', $request->full_mobile);
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to send OTP: ' . $e->getMessage()]);
        }
    }
}